/**
* \file: AlsaAudioSourceImpl.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Android Auto
*
* \author: I. Hayashi / ADITJ/SW / ihayashi@jp.adit-jv.com
*
* \copyright (c) 2014-2015 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/
#ifndef ALSAAUDIOSOURCEIMPL_H
#define ALSAAUDIOSOURCEIMPL_H

#include <aauto/AudioSource.h>
#include <aauto/GalReceiver.h>
#include <aauto/util/scoped_ptr.h>
#include <aauto/util/shared_ptr.h>
#include <aauto/util/Thread.h>

#include "aauto/AditAudioSource.h"
#include <uspi/ConfigHandle.h>
#include "AlsaAudioCommon.h"

#define NSEC_PER_SEC                (1000ULL * 1000ULL * 1000ULL)

#define MICROPHONE_WORTH_OF_SAMPLE_MS   10
#define MICROPHONE_SAMPLING_RATE        16000
#define MICROPHONE_BITS_PER_SAMPLE      16
#define MICROPHONE_CHANNELS             1
// Macro replaced by perioed aligned calculation
#define MICROPHONE_FRAGSIZE             ((MICROPHONE_SAMPLING_RATE * (MICROPHONE_BITS_PER_SAMPLE / 8) \
                                          * MICROPHONE_CHANNELS) * MICROPHONE_WORTH_OF_SAMPLE_MS / 1000)

namespace adit { namespace aauto
{
class AlsaSourceConfig : public uspi::ConfigHandle
{
public:
AlsaSourceConfig()
{
    mSampleRate = 0;
    mInitToutms = 0;
    mNumBits = 0;
    mNumChannels = 0;
    mVerbose = 0;
    mAlsaLogging = 0;
    mDisablePrio = 0;
    mThreadPrio = 0;
    mPeriodms = 0;
    mBufferperiods = 0;
    mSilencems = 0;
    mDevice = "";
}

bool ResultConfig()
{
    bool isValid = true;
    mSampleRate     = getInt("audio-source-sampling-rate", &isValid, Range(-1, INT_MAX), MICROPHONE_SAMPLING_RATE);
    mNumBits        = getInt("audio-source-bits-per-sample", &isValid, Range(-1, INT_MAX), MICROPHONE_BITS_PER_SAMPLE);
    mNumChannels    = getInt("audio-source-channels", &isValid, Range(-1, INT_MAX), MICROPHONE_CHANNELS);
    mVerbose        = getInt("enable-verbose-logging", &isValid, Range(0, 1), 0);
    mAlsaLogging    = getInt("alsa-dump-configuration", &isValid, Range(0, 1), 0);
    mDisablePrio    = getInt("disable-real-time-priority-audio", &isValid, Range(0, 1), 0);
    mThreadPrio     = getInt("audio-threads-real-time-priority", &isValid, Range(-1, INT_MAX), 61);
    mPeriodms       = getIntMultiKey("alsa-main-audio-in", "periodms", &isValid, Range(-1, INT_MAX));
    mBufferperiods  = getIntMultiKey("alsa-main-audio-in", "bufferperiods", &isValid, Range(-1, INT_MAX), -1);
    mSilencems      = getIntMultiKey("alsa-main-audio-in", "silencems", &isValid, Range(-1, INT_MAX), -1);
    mInitToutms     = getIntMultiKey("alsa-main-audio-in", "inittoutms", &isValid, Range(-1, INT_MAX), -1);
    mDevice         = getStringMultiKey("alsa-main-audio-in", "device", &isValid, Range(-1, INT_MAX));

    return isValid;
}
    int     mSampleRate;        // Sampling rate
    int     mNumBits;           // Bits per sample
    int     mNumChannels;       // Number of Channels

    bool    mVerbose;           // Verbose log flag 
    bool    mAlsaLogging;       // Alsa log flag 
    bool    mDisablePrio;       // Priority Set flag
    int     mThreadPrio;        // Thread Priority
    string  mAlsaSourceSetting; // Alsa Common settings data

    // ALSA device setting
    int     mInitToutms;
    int     mPeriodms;
    int     mBufferperiods;
    int     mSilencems;
    string  mDevice;
};

class AlsaAudioSourceImpl : public IAudioSourceCallbacks
{
public:
    AlsaAudioSourceImpl(AudioSource* inSource);
    ~AlsaAudioSourceImpl();
    bool init();
    void shutdown();

    void setConfigItem(string inKey, string inValue);
    void registerCallbacks(IAditAudioSourceCallbacks* inCallbacks);

protected:
    /* IAudioSourceCallbacks */
    void ackCallback(int sessionId, uint32_t ack);
    int microphoneRequestCallback(bool open, bool ancEnabled,
            bool ecEnabled, int maxUnacked);

private:
    class MicrophoneReaderThread : public Thread
    {
    public:
        MicrophoneReaderThread(AlsaAudioSourceImpl* audioSource) : mAudioSource(audioSource) { }
    protected:
        // Thread implementation
        void run();
    private:
        uint64_t getTimestamp();
        AlsaAudioSourceImpl* mAudioSource;
    };

    bool mStarted;
    bool mRunning;
    int mSessionId;
    string mSourceName;

    IAditAudioSourceCallbacks* mCallbacks;
    AudioSource* audioSource;
    scoped_ptr<AlsaAudioCommon> mRecorder;
    scoped_ptr<MicrophoneReaderThread> mReaderThread;
    scoped_ptr<Semaphore> mSem;

    /**
     * @brief  Helper API to get the current configuration
     *         and set the internal members (e.g.mSampleRate, etc.)
     */
    bool getCurrConfig(void);
    /**
     * @brief Helper API, called in case of microphoneRequestCallback(open==true) from MD.
     *        Prepares ALSA device and starts MicrophoneReaderThread.
     */
    bool start(int maxUnacked);
    /**
     * @brief Helper API, called in case of microphoneRequestCallback(open==false) from MD
     *        or shutdown triggered by Application.
     *        Stop and release ALSA device and joins the MicrophoneReaderThread.
     */
    void stop();
    /**
     * @brief Helper API, called by helper function start().
     *        Set up the ALSA device by using the configuration set by Application.
     */
    bool prepare();

    AlsaSourceConfig  mConfig;
};
} } /* namespace adit { namespace aauto */

#endif // ALSAAUDIOSOURCEIMPL_H
